﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Threading;
using System.Windows.Forms;

namespace PWS.API.Keyboard.Sample10
{
    public partial class FormMain : Form
    {
        [DllImport("user32.dll")]
        public static extern UInt32 SendInput(UInt32 numberOfInputs, Input[] inputs, Int32 sizeOfInputStructure);

        [DllImport("user32.dll")]
        public static extern IntPtr FindWindow(string lpClassName, string lpWindowName);

        [DllImport("user32.dll")]
        public static extern bool SetForegroundWindow(IntPtr hWnd);

        #region Struct Declaration

        public struct Input
        {
            public UInt32 Type;

            public MouseKeyboardHardwareInput Data;

            public static Int32 SizeOf
            {
                get { return Marshal.SizeOf(typeof(Input)); }
            }
        }

        [StructLayout(LayoutKind.Explicit)]
        public struct MouseKeyboardHardwareInput
        {
            [FieldOffset(0)]
            public MouseInput Mouse;

            [FieldOffset(0)]
            public KeyboardInput Keyboard;

            [FieldOffset(0)]
            public HardwareInput Hardware;
        }

        public enum InputType : uint // UInt32
        {
            Mouse = 0,
            Keyboard = 1,
            Hardware = 2,
        }

        public struct MouseInput
        {
            public Int32 X;
            public Int32 Y;
            public UInt32 MouseData;
            public UInt32 Flags;
            public UInt32 Time;
            public IntPtr ExtraInfo;
        }

        public struct KeyboardInput
        {
            public UInt16 KeyCode;
            public UInt16 Scan;
            public UInt32 Flags;
            public UInt32 Time;
            public IntPtr ExtraInfo;
        }

        public struct HardwareInput
        {
            public UInt32 Msg;
            public UInt16 ParamL;
            public UInt16 ParamH;
        }

        [Flags]
        public enum KeyboardFlag : uint // UInt32
        {
            ExtendedKey = 0x0001,
            KeyUp = 0x0002,
            Unicode = 0x0004,
            ScanCode = 0x0008,
        }

        public enum VirtualKeyCode : ushort // UInt16
        {
            LBUTTON = 0x01,

            RBUTTON = 0x02,

            CANCEL = 0x03,

            /// <summary>
            /// Middle mouse button (three-button mouse) - NOT contiguous with LBUTTON and RBUTTON
            /// </summary>
            MBUTTON = 0x04,

            /// <summary>
            /// Windows 2000/XP: X1 mouse button - NOT contiguous with LBUTTON and RBUTTON
            /// </summary>
            XBUTTON1 = 0x05,

            /// <summary>
            /// Windows 2000/XP: X2 mouse button - NOT contiguous with LBUTTON and RBUTTON
            /// </summary>
            XBUTTON2 = 0x06,

            // 0x07 : Undefined

            /// <summary>
            /// BACKSPACE key
            /// </summary>
            BACK = 0x08,

            /// <summary>
            /// TAB key
            /// </summary>
            TAB = 0x09,

            // 0x0A - 0x0B : Reserved

            /// <summary>
            /// CLEAR key
            /// </summary>
            CLEAR = 0x0C,

            /// <summary>
            /// ENTER key
            /// </summary>
            RETURN = 0x0D,

            // 0x0E - 0x0F : Undefined

            /// <summary>
            /// SHIFT key
            /// </summary>
            SHIFT = 0x10,

            /// <summary>
            /// CTRL key
            /// </summary>
            CONTROL = 0x11,

            /// <summary>
            /// ALT key
            /// </summary>
            MENU = 0x12,

            /// <summary>
            /// PAUSE key
            /// </summary>
            PAUSE = 0x13,

            /// <summary>
            /// CAPS LOCK key
            /// </summary>
            CAPITAL = 0x14,

            /// <summary>
            /// Input Method Editor (IME) Kana mode
            /// </summary>
            KANA = 0x15,

            /// <summary>
            /// IME Hanguel mode (maintained for compatibility; use HANGUL)
            /// </summary>
            HANGEUL = 0x15,

            /// <summary>
            /// IME Hangul mode
            /// </summary>
            HANGUL = 0x15,

            // 0x16 : Undefined

            /// <summary>
            /// IME Junja mode
            /// </summary>
            JUNJA = 0x17,

            /// <summary>
            /// IME final mode
            /// </summary>
            FINAL = 0x18,

            /// <summary>
            /// IME Hanja mode
            /// </summary>
            HANJA = 0x19,

            /// <summary>
            /// IME Kanji mode
            /// </summary>
            KANJI = 0x19,

            // 0x1A : Undefined

            /// <summary>
            /// ESC key
            /// </summary>
            ESCAPE = 0x1B,

            /// <summary>
            /// IME convert
            /// </summary>
            CONVERT = 0x1C,

            /// <summary>
            /// IME nonconvert
            /// </summary>
            NONCONVERT = 0x1D,

            /// <summary>
            /// IME accept
            /// </summary>
            ACCEPT = 0x1E,

            /// <summary>
            /// IME mode change request
            /// </summary>
            MODECHANGE = 0x1F,

            /// <summary>
            /// SPACEBAR
            /// </summary>
            SPACE = 0x20,

            /// <summary>
            /// PAGE UP key
            /// </summary>
            PRIOR = 0x21,

            /// <summary>
            /// PAGE DOWN key
            /// </summary>
            NEXT = 0x22,

            /// <summary>
            /// END key
            /// </summary>
            END = 0x23,

            /// <summary>
            /// HOME key
            /// </summary>
            HOME = 0x24,

            /// <summary>
            /// LEFT ARROW key
            /// </summary>
            LEFT = 0x25,

            /// <summary>
            /// UP ARROW key
            /// </summary>
            UP = 0x26,

            /// <summary>
            /// RIGHT ARROW key
            /// </summary>
            RIGHT = 0x27,

            /// <summary>
            /// DOWN ARROW key
            /// </summary>
            DOWN = 0x28,

            /// <summary>
            /// SELECT key
            /// </summary>
            SELECT = 0x29,

            /// <summary>
            /// PRINT key
            /// </summary>
            PRINT = 0x2A,

            /// <summary>
            /// EXECUTE key
            /// </summary>
            EXECUTE = 0x2B,

            /// <summary>
            /// PRINT SCREEN key
            /// </summary>
            SNAPSHOT = 0x2C,

            /// <summary>
            /// INS key
            /// </summary>
            INSERT = 0x2D,

            /// <summary>
            /// DEL key
            /// </summary>
            DELETE = 0x2E,

            /// <summary>
            /// HELP key
            /// </summary>
            HELP = 0x2F,

            /// <summary>
            /// 0 key
            /// </summary>
            VK_0 = 0x30,

            /// <summary>
            /// 1 key
            /// </summary>
            VK_1 = 0x31,

            /// <summary>
            /// 2 key
            /// </summary>
            VK_2 = 0x32,

            /// <summary>
            /// 3 key
            /// </summary>
            VK_3 = 0x33,

            /// <summary>
            /// 4 key
            /// </summary>
            VK_4 = 0x34,

            /// <summary>
            /// 5 key
            /// </summary>
            VK_5 = 0x35,

            /// <summary>
            /// 6 key
            /// </summary>
            VK_6 = 0x36,

            /// <summary>
            /// 7 key
            /// </summary>
            VK_7 = 0x37,

            /// <summary>
            /// 8 key
            /// </summary>
            VK_8 = 0x38,

            /// <summary>
            /// 9 key
            /// </summary>
            VK_9 = 0x39,

            //
            // 0x3A - 0x40 : Udefined
            //

            /// <summary>
            /// A key
            /// </summary>
            VK_A = 0x41,

            /// <summary>
            /// B key
            /// </summary>
            VK_B = 0x42,

            /// <summary>
            /// C key
            /// </summary>
            VK_C = 0x43,

            /// <summary>
            /// D key
            /// </summary>
            VK_D = 0x44,

            /// <summary>
            /// E key
            /// </summary>
            VK_E = 0x45,

            /// <summary>
            /// F key
            /// </summary>
            VK_F = 0x46,

            /// <summary>
            /// G key
            /// </summary>
            VK_G = 0x47,

            /// <summary>
            /// H key
            /// </summary>
            VK_H = 0x48,

            /// <summary>
            /// I key
            /// </summary>
            VK_I = 0x49,

            /// <summary>
            /// J key
            /// </summary>
            VK_J = 0x4A,

            /// <summary>
            /// K key
            /// </summary>
            VK_K = 0x4B,

            /// <summary>
            /// L key
            /// </summary>
            VK_L = 0x4C,

            /// <summary>
            /// M key
            /// </summary>
            VK_M = 0x4D,

            /// <summary>
            /// N key
            /// </summary>
            VK_N = 0x4E,

            /// <summary>
            /// O key
            /// </summary>
            VK_O = 0x4F,

            /// <summary>
            /// P key
            /// </summary>
            VK_P = 0x50,

            /// <summary>
            /// Q key
            /// </summary>
            VK_Q = 0x51,

            /// <summary>
            /// R key
            /// </summary>
            VK_R = 0x52,

            /// <summary>
            /// S key
            /// </summary>
            VK_S = 0x53,

            /// <summary>
            /// T key
            /// </summary>
            VK_T = 0x54,

            /// <summary>
            /// U key
            /// </summary>
            VK_U = 0x55,

            /// <summary>
            /// V key
            /// </summary>
            VK_V = 0x56,

            /// <summary>
            /// W key
            /// </summary>
            VK_W = 0x57,

            /// <summary>
            /// X key
            /// </summary>
            VK_X = 0x58,

            /// <summary>
            /// Y key
            /// </summary>
            VK_Y = 0x59,

            /// <summary>
            /// Z key
            /// </summary>
            VK_Z = 0x5A,

            /// <summary>
            /// Left Windows key (Microsoft Natural keyboard)
            /// </summary>
            LWIN = 0x5B,

            /// <summary>
            /// Right Windows key (Natural keyboard)
            /// </summary>
            RWIN = 0x5C,

            /// <summary>
            /// Applications key (Natural keyboard)
            /// </summary>
            APPS = 0x5D,

            // 0x5E : reserved

            /// <summary>
            /// Computer Sleep key
            /// </summary>
            SLEEP = 0x5F,

            /// <summary>
            /// Numeric keypad 0 key
            /// </summary>
            NUMPAD0 = 0x60,

            /// <summary>
            /// Numeric keypad 1 key
            /// </summary>
            NUMPAD1 = 0x61,

            /// <summary>
            /// Numeric keypad 2 key
            /// </summary>
            NUMPAD2 = 0x62,

            /// <summary>
            /// Numeric keypad 3 key
            /// </summary>
            NUMPAD3 = 0x63,

            /// <summary>
            /// Numeric keypad 4 key
            /// </summary>
            NUMPAD4 = 0x64,

            /// <summary>
            /// Numeric keypad 5 key
            /// </summary>
            NUMPAD5 = 0x65,

            /// <summary>
            /// Numeric keypad 6 key
            /// </summary>
            NUMPAD6 = 0x66,

            /// <summary>
            /// Numeric keypad 7 key
            /// </summary>
            NUMPAD7 = 0x67,

            /// <summary>
            /// Numeric keypad 8 key
            /// </summary>
            NUMPAD8 = 0x68,

            /// <summary>
            /// Numeric keypad 9 key
            /// </summary>
            NUMPAD9 = 0x69,

            /// <summary>
            /// Multiply key
            /// </summary>
            MULTIPLY = 0x6A,

            /// <summary>
            /// Add key
            /// </summary>
            ADD = 0x6B,

            /// <summary>
            /// Separator key
            /// </summary>
            SEPARATOR = 0x6C,

            /// <summary>
            /// Subtract key
            /// </summary>
            SUBTRACT = 0x6D,

            /// <summary>
            /// Decimal key
            /// </summary>
            DECIMAL = 0x6E,

            /// <summary>
            /// Divide key
            /// </summary>
            DIVIDE = 0x6F,

            /// <summary>
            /// F1 key
            /// </summary>
            F1 = 0x70,

            /// <summary>
            /// F2 key
            /// </summary>
            F2 = 0x71,

            /// <summary>
            /// F3 key
            /// </summary>
            F3 = 0x72,

            /// <summary>
            /// F4 key
            /// </summary>
            F4 = 0x73,

            /// <summary>
            /// F5 key
            /// </summary>
            F5 = 0x74,

            /// <summary>
            /// F6 key
            /// </summary>
            F6 = 0x75,

            /// <summary>
            /// F7 key
            /// </summary>
            F7 = 0x76,

            /// <summary>
            /// F8 key
            /// </summary>
            F8 = 0x77,

            /// <summary>
            /// F9 key
            /// </summary>
            F9 = 0x78,

            /// <summary>
            /// F10 key
            /// </summary>
            F10 = 0x79,

            /// <summary>
            /// F11 key
            /// </summary>
            F11 = 0x7A,

            /// <summary>
            /// F12 key
            /// </summary>
            F12 = 0x7B,

            /// <summary>
            /// F13 key
            /// </summary>
            F13 = 0x7C,

            /// <summary>
            /// F14 key
            /// </summary>
            F14 = 0x7D,

            /// <summary>
            /// F15 key
            /// </summary>
            F15 = 0x7E,

            /// <summary>
            /// F16 key
            /// </summary>
            F16 = 0x7F,

            /// <summary>
            /// F17 key
            /// </summary>
            F17 = 0x80,

            /// <summary>
            /// F18 key
            /// </summary>
            F18 = 0x81,

            /// <summary>
            /// F19 key
            /// </summary>
            F19 = 0x82,

            /// <summary>
            /// F20 key
            /// </summary>
            F20 = 0x83,

            /// <summary>
            /// F21 key
            /// </summary>
            F21 = 0x84,

            /// <summary>
            /// F22 key
            /// </summary>
            F22 = 0x85,

            /// <summary>
            /// F23 key
            /// </summary>
            F23 = 0x86,

            /// <summary>
            /// F24 key
            /// </summary>
            F24 = 0x87,

            //
            // 0x88 - 0x8F : Unassigned
            //

            /// <summary>
            /// NUM LOCK key
            /// </summary>
            NUMLOCK = 0x90,

            /// <summary>
            /// SCROLL LOCK key
            /// </summary>
            SCROLL = 0x91,

            // 0x92 - 0x96 : OEM Specific

            // 0x97 - 0x9F : Unassigned

            //
            // L* & R* - left and right Alt, Ctrl and Shift virtual keys.
            // Used only as parameters to GetAsyncKeyState() and GetKeyState().
            // No other API or message will distinguish left and right keys in this way.
            //

            /// <summary>
            /// Left SHIFT key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            LSHIFT = 0xA0,

            /// <summary>
            /// Right SHIFT key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            RSHIFT = 0xA1,

            /// <summary>
            /// Left CONTROL key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            LCONTROL = 0xA2,

            /// <summary>
            /// Right CONTROL key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            RCONTROL = 0xA3,

            /// <summary>
            /// Left MENU key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            LMENU = 0xA4,

            /// <summary>
            /// Right MENU key - Used only as parameters to GetAsyncKeyState() and GetKeyState()
            /// </summary>
            RMENU = 0xA5,

            /// <summary>
            /// Windows 2000/XP: Browser Back key
            /// </summary>
            BROWSER_BACK = 0xA6,

            /// <summary>
            /// Windows 2000/XP: Browser Forward key
            /// </summary>
            BROWSER_FORWARD = 0xA7,

            /// <summary>
            /// Windows 2000/XP: Browser Refresh key
            /// </summary>
            BROWSER_REFRESH = 0xA8,

            /// <summary>
            /// Windows 2000/XP: Browser Stop key
            /// </summary>
            BROWSER_STOP = 0xA9,

            /// <summary>
            /// Windows 2000/XP: Browser Search key
            /// </summary>
            BROWSER_SEARCH = 0xAA,

            /// <summary>
            /// Windows 2000/XP: Browser Favorites key
            /// </summary>
            BROWSER_FAVORITES = 0xAB,

            /// <summary>
            /// Windows 2000/XP: Browser Start and Home key
            /// </summary>
            BROWSER_HOME = 0xAC,

            /// <summary>
            /// Windows 2000/XP: Volume Mute key
            /// </summary>
            VOLUME_MUTE = 0xAD,

            /// <summary>
            /// Windows 2000/XP: Volume Down key
            /// </summary>
            VOLUME_DOWN = 0xAE,

            /// <summary>
            /// Windows 2000/XP: Volume Up key
            /// </summary>
            VOLUME_UP = 0xAF,

            /// <summary>
            /// Windows 2000/XP: Next Track key
            /// </summary>
            MEDIA_NEXT_TRACK = 0xB0,

            /// <summary>
            /// Windows 2000/XP: Previous Track key
            /// </summary>
            MEDIA_PREV_TRACK = 0xB1,

            /// <summary>
            /// Windows 2000/XP: Stop Media key
            /// </summary>
            MEDIA_STOP = 0xB2,

            /// <summary>
            /// Windows 2000/XP: Play/Pause Media key
            /// </summary>
            MEDIA_PLAY_PAUSE = 0xB3,

            /// <summary>
            /// Windows 2000/XP: Start Mail key
            /// </summary>
            LAUNCH_MAIL = 0xB4,

            /// <summary>
            /// Windows 2000/XP: Select Media key
            /// </summary>
            LAUNCH_MEDIA_SELECT = 0xB5,

            /// <summary>
            /// Windows 2000/XP: Start Application 1 key
            /// </summary>
            LAUNCH_APP1 = 0xB6,

            /// <summary>
            /// Windows 2000/XP: Start Application 2 key
            /// </summary>
            LAUNCH_APP2 = 0xB7,

            //
            // 0xB8 - 0xB9 : Reserved
            //

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the ';:' key
            /// </summary>
            OEM_1 = 0xBA,

            /// <summary>
            /// Windows 2000/XP: For any country/region, the '+' key
            /// </summary>
            OEM_PLUS = 0xBB,

            /// <summary>
            /// Windows 2000/XP: For any country/region, the ',' key
            /// </summary>
            OEM_COMMA = 0xBC,

            /// <summary>
            /// Windows 2000/XP: For any country/region, the '-' key
            /// </summary>
            OEM_MINUS = 0xBD,

            /// <summary>
            /// Windows 2000/XP: For any country/region, the '.' key
            /// </summary>
            OEM_PERIOD = 0xBE,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the '/?' key
            /// </summary>
            OEM_2 = 0xBF,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the '`~' key
            /// </summary>
            OEM_3 = 0xC0,

            //
            // 0xC1 - 0xD7 : Reserved
            //

            //
            // 0xD8 - 0xDA : Unassigned
            //

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the '[{' key
            /// </summary>
            OEM_4 = 0xDB,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the '\|' key
            /// </summary>
            OEM_5 = 0xDC,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the ']}' key
            /// </summary>
            OEM_6 = 0xDD,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard. Windows 2000/XP: For the US standard keyboard, the 'single-quote/double-quote' key
            /// </summary>
            OEM_7 = 0xDE,

            /// <summary>
            /// Used for miscellaneous characters; it can vary by keyboard.
            /// </summary>
            OEM_8 = 0xDF,

            //
            // 0xE0 : Reserved
            //

            //
            // 0xE1 : OEM Specific
            //

            /// <summary>
            /// Windows 2000/XP: Either the angle bracket key or the backslash key on the RT 102-key keyboard
            /// </summary>
            OEM_102 = 0xE2,

            //
            // (0xE3-E4) : OEM specific
            //

            /// <summary>
            /// Windows 95/98/Me, Windows NT 4.0, Windows 2000/XP: IME PROCESS key
            /// </summary>
            PROCESSKEY = 0xE5,

            //
            // 0xE6 : OEM specific
            //

            /// <summary>
            /// Windows 2000/XP: Used to pass Unicode characters as if they were keystrokes. The PACKET key is the low word of a 32-bit Virtual Key value used for non-keyboard input methods. For more information, see Remark in KEYBDINPUT, SendInput, WM_KEYDOWN, and WM_KEYUP
            /// </summary>
            PACKET = 0xE7,

            //
            // 0xE8 : Unassigned
            //

            //
            // 0xE9-F5 : OEM specific
            //

            /// <summary>
            /// Attn key
            /// </summary>
            ATTN = 0xF6,

            /// <summary>
            /// CrSel key
            /// </summary>
            CRSEL = 0xF7,

            /// <summary>
            /// ExSel key
            /// </summary>
            EXSEL = 0xF8,

            /// <summary>
            /// Erase EOF key
            /// </summary>
            EREOF = 0xF9,

            /// <summary>
            /// Play key
            /// </summary>
            PLAY = 0xFA,

            /// <summary>
            /// Zoom key
            /// </summary>
            ZOOM = 0xFB,

            /// <summary>
            /// Reserved
            /// </summary>
            NONAME = 0xFC,

            /// <summary>
            /// PA1 key
            /// </summary>
            PA1 = 0xFD,

            /// <summary>
            /// Clear key
            /// </summary>
            OEM_CLEAR = 0xFE,
        }

        [Flags]
        public enum MouseFlag : uint // UInt32
        {
            Move = 0x0001,
            LeftDown = 0x0002,
            LeftUp = 0x0004,
            RightDown = 0x0008,
            RightUp = 0x0010,
            MiddleDown = 0x0020,
            MiddleUp = 0x0040,
            XDown = 0x0080,
            XUp = 0x0100,
            VerticalWheel = 0x0800,
            HorizontalWheel = 0x1000,
            VirtualDesk = 0x4000,
            Absolute = 0x8000,
        }

        public enum MouseButton
        {
            LeftButton,
            MiddleButton,
            RightButton,
        }

        public static MouseFlag ToMouseButtonDownFlag(MouseButton button)
        {
            switch (button)
            {
                case MouseButton.LeftButton:
                    return MouseFlag.LeftDown;

                case MouseButton.MiddleButton:
                    return MouseFlag.MiddleDown;

                case MouseButton.RightButton:
                    return MouseFlag.RightDown;

                default:
                    return MouseFlag.LeftDown;
            }
        }

        public static MouseFlag ToMouseButtonUpFlag(MouseButton button)
        {
            switch (button)
            {
                case MouseButton.LeftButton:
                    return MouseFlag.LeftUp;

                case MouseButton.MiddleButton:
                    return MouseFlag.MiddleUp;

                case MouseButton.RightButton:
                    return MouseFlag.RightUp;

                default:
                    return MouseFlag.LeftUp;
            }
        }

        #endregion Struct Declaration

        #region Keyboard Function

        public void AddKeyDown(VirtualKeyCode keyCode)
        {
            var down = new Input();
            down.Type = (UInt32)InputType.Keyboard;
            down.Data.Keyboard = new KeyboardInput();
            down.Data.Keyboard.KeyCode = (UInt16)keyCode;
            down.Data.Keyboard.Scan = 0;
            down.Data.Keyboard.Flags = 0;
            down.Data.Keyboard.Time = 0;
            down.Data.Keyboard.ExtraInfo = IntPtr.Zero;

            inputList.Add(down);
        }

        public void AddKeyUp(VirtualKeyCode keyCode)
        {
            var up = new Input();
            up.Type = (UInt32)InputType.Keyboard;
            up.Data.Keyboard = new KeyboardInput();
            up.Data.Keyboard.KeyCode = (UInt16)keyCode;
            up.Data.Keyboard.Scan = 0;
            up.Data.Keyboard.Flags = (UInt32)KeyboardFlag.KeyUp;
            up.Data.Keyboard.Time = 0;
            up.Data.Keyboard.ExtraInfo = IntPtr.Zero;

            inputList.Add(up);
        }

        public void AddKeyPress(VirtualKeyCode keyCode)
        {
            AddKeyDown(keyCode);
            AddKeyUp(keyCode);
        }

        public void AddCharacter(char character)
        {
            UInt16 scanCode = character;

            var down = new Input();
            down.Type = (UInt32)InputType.Keyboard;
            down.Data.Keyboard = new KeyboardInput();
            down.Data.Keyboard.KeyCode = 0;
            down.Data.Keyboard.Scan = scanCode;
            down.Data.Keyboard.Flags = (UInt32)KeyboardFlag.Unicode;
            down.Data.Keyboard.Time = 0;
            down.Data.Keyboard.ExtraInfo = IntPtr.Zero;

            var up = new Input();
            up.Type = (UInt32)InputType.Keyboard;
            up.Data.Keyboard = new KeyboardInput();
            up.Data.Keyboard.KeyCode = 0;
            up.Data.Keyboard.Scan = scanCode;
            up.Data.Keyboard.Flags = (UInt32)(KeyboardFlag.KeyUp | KeyboardFlag.Unicode);
            up.Data.Keyboard.Time = 0;
            up.Data.Keyboard.ExtraInfo = IntPtr.Zero;

            // Handle extended keys:
            // If the scan code is preceded by a prefix byte that has the value 0xE0 (224),
            // we need to include the KEYEVENTF_EXTENDEDKEY flag in the Flags property.
            if ((scanCode & 0xFF00) == 0xE000)
            {
                down.Data.Keyboard.Flags |= (UInt32)KeyboardFlag.ExtendedKey;
                up.Data.Keyboard.Flags |= (UInt32)KeyboardFlag.ExtendedKey;
            }

            inputList.Add(down);
            inputList.Add(up);
        }

        public void AddCharacters(IEnumerable<char> characters)
        {
            foreach (var character in characters)
            {
                AddCharacter(character);
            }
        }

        public void AddCharacters(string characters)
        {
            AddCharacters(characters.ToCharArray());
        }

        #endregion Keyboard Function

        #region Mouse Function

        public void AddRelativeMouseMovement(int x, int y)
        {
            var movement = new Input();
            movement.Type = (UInt32)InputType.Mouse;
            movement.Data.Mouse.Flags = (UInt32)MouseFlag.Move;
            movement.Data.Mouse.X = x;
            movement.Data.Mouse.Y = y;

            inputList.Add(movement);
        }

        public void AddAbsoluteMouseMovement(int absoluteX, int absoluteY)
        {
            var movement = new Input();
            movement.Type = (UInt32)InputType.Mouse;
            movement.Data.Mouse.Flags = (UInt32)(MouseFlag.Move | MouseFlag.Absolute);
            movement.Data.Mouse.X = absoluteX;
            movement.Data.Mouse.Y = absoluteY;

            inputList.Add(movement);
        }

        public void AddAbsoluteMouseMovementOnVirtualDesktop(int absoluteX, int absoluteY)
        {
            var movement = new Input();
            movement.Type = (UInt32)InputType.Mouse;
            movement.Data.Mouse.Flags = (UInt32)(MouseFlag.Move | MouseFlag.Absolute | MouseFlag.VirtualDesk);
            movement.Data.Mouse.X = absoluteX;
            movement.Data.Mouse.Y = absoluteY;

            inputList.Add(movement);
        }

        public void AddMouseButtonDown(MouseButton button)
        {
            var buttonDown = new Input();
            buttonDown.Type = (UInt32)InputType.Mouse;
            buttonDown.Data.Mouse.Flags = (UInt32)ToMouseButtonDownFlag(button);

            inputList.Add(buttonDown);
        }

        public void AddMouseButtonUp(MouseButton button)
        {
            var buttonUp = new Input();
            buttonUp.Type = (UInt32)InputType.Mouse;
            buttonUp.Data.Mouse.Flags = (UInt32)ToMouseButtonUpFlag(button);
            inputList.Add(buttonUp);
        }

        public void AddMouseButtonDoubleClick(MouseButton button)
        {
            AddMouseButtonClick(button);
            AddMouseButtonClick(button);
        }

        public void AddMouseButtonClick(MouseButton button)
        {
            AddMouseButtonDown(button);
            AddMouseButtonUp(button);
        }

        #endregion Mouse Function

        private List<Input> inputList;

        public FormMain()
        {
            InitializeComponent();
        }

        private void buttonRightClickSimulation_Click(object sender, EventArgs e)
        {
            inputList = new List<Input>();
            var hWindow = FindWindow("notepad", null);
            SetForegroundWindow(hWindow);
            Thread.Sleep(500);
            AddRelativeMouseMovement(500, 500);
            AddKeyPress(VirtualKeyCode.CONTROL);
            //AddMouseButtonClick(MouseButton.RightButton);
            SendInput((UInt32)inputList.Count, inputList.ToArray(), Input.SizeOf);
        }

        private void buttonKeyboardSimulation_Click(object sender, EventArgs e)
        {
            inputList = new List<Input>();
            var hWindow = FindWindow("notepad", null);
            SetForegroundWindow(hWindow);
            Thread.Sleep(500);
            AddCharacters("صابر فتح اللهی");
            AddKeyPress(VirtualKeyCode.RETURN);
            AddCharacters("khobi?");
            AddKeyPress(VirtualKeyCode.RETURN);
            AddCharacter('a');
            AddKeyPress(VirtualKeyCode.CAPITAL);
            SendInput((UInt32)inputList.Count, inputList.ToArray(), Input.SizeOf);
        }
    }
}